/*
 * Decompiled with CFR 0.152.
 */
package com.akaene.eccairs;

import com.akaene.eccairs.EccairsAttribute;
import com.akaene.eccairs.EccairsEntity;
import com.akaene.eccairs.EccairsValue;
import com.akaene.eccairs.TaxonomyServiceException;
import com.akaene.eccairs.TaxonomyServiceResponse;
import com.akaene.eccairs.TaxonomyVersionInfo;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.PathNotFoundException;
import com.jayway.jsonpath.Predicate;
import com.jayway.jsonpath.TypeRef;
import com.jayway.jsonpath.spi.json.JacksonJsonProvider;
import com.jayway.jsonpath.spi.json.JsonProvider;
import com.jayway.jsonpath.spi.mapper.JacksonMappingProvider;
import com.jayway.jsonpath.spi.mapper.MappingProvider;
import java.io.IOException;
import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EccairsTaxonomyService {
    private static final Logger LOG = LoggerFactory.getLogger(EccairsTaxonomyService.class);
    private static final int MAX_ATTEMPTS = 5;
    private final String taxonomyServiceUrl;
    private TaxonomyVersionInfo taxonomyVersion;
    private DocumentContext taxonomyTree;
    private final ObjectMapper objectMapper;
    private final HttpClient httpClient;
    private final Map<Integer, Integer> attributeIdMap = new HashMap<Integer, Integer>();

    public EccairsTaxonomyService(String taxonomyServiceUrl) {
        if (taxonomyServiceUrl == null || taxonomyServiceUrl.isBlank()) {
            throw new IllegalArgumentException("Taxonomy service '" + taxonomyServiceUrl + "' URL is not valid.");
        }
        this.taxonomyServiceUrl = Objects.requireNonNull(taxonomyServiceUrl);
        this.httpClient = HttpClient.newHttpClient();
        this.objectMapper = new ObjectMapper();
        EccairsTaxonomyService.configureJsonPath();
    }

    public String getTaxonomyVersion() {
        this.initializeIfNecessary();
        return this.taxonomyVersion.label();
    }

    public int getTaxonomyVersionId() {
        this.initializeIfNecessary();
        return this.taxonomyVersion.id();
    }

    private void initializeIfNecessary() {
        if (this.taxonomyVersion != null) {
            return;
        }
        LOG.debug("Initializing ECCAIRS taxonomy service.");
        this.taxonomyVersion = this.loadTaxonomyVersionInfo();
        LOG.debug("Current taxonomy: {} (internal ECCAIRS ID: {})", (Object)this.taxonomyVersion.label(), (Object)this.taxonomyVersion.id());
        this.taxonomyTree = this.loadTaxonomyTree();
    }

    public TaxonomyVersionInfo loadTaxonomyVersionInfo() {
        TaxonomyServiceResponse versionInfo = this.getResponse(this.taxonomyServiceUrl + "/version/public");
        assert (versionInfo != null);
        DocumentContext node = JsonPath.parse((String)versionInfo.getData().toString());
        int versionId = (Integer)node.read("$.id", Integer.class, new Predicate[0]);
        String versionLabel = (String)node.read("$.version", String.class, new Predicate[0]);
        return new TaxonomyVersionInfo(versionLabel, versionId);
    }

    private DocumentContext loadTaxonomyTree() {
        TaxonomyServiceResponse tree = this.getResponse(this.taxonomyServiceUrl + "/tree/public");
        assert (tree != null);
        return JsonPath.parse((String)tree.getData().toString());
    }

    private TaxonomyServiceResponse getResponse(String uri) {
        try {
            HttpRequest request = HttpRequest.newBuilder().GET().uri(URI.create(uri)).header("Accept", "application/json").build();
            return this.attemptRequest(request, 0);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new TaxonomyServiceException("Unable to perform request.", e);
        }
    }

    private TaxonomyServiceResponse attemptRequest(HttpRequest request, int attempt) throws InterruptedException {
        try {
            HttpResponse<String> resp = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
            if (resp.statusCode() != 200) {
                LOG.error("Failed to get response. Received {}.", resp);
                throw new TaxonomyServiceException("Unable to retrieve response. Got status " + resp.statusCode());
            }
            return (TaxonomyServiceResponse)this.objectMapper.readValue(resp.body(), TaxonomyServiceResponse.class);
        }
        catch (IOException | RuntimeException e) {
            if (e.getCause() instanceof ConnectException && attempt <= 5) {
                LOG.warn("Failed to get response due to {}. Attempting again in 10s.", (Object)e.getMessage());
                Thread.sleep(10000L);
                return this.attemptRequest(request, attempt + 1);
            }
            throw new TaxonomyServiceException("Unable to get response.", e);
        }
    }

    public boolean hasHierarchicalValueList(int attributeId) {
        LOG.trace("Checking hierarchy of value list of attribute {}.", (Object)attributeId);
        this.initializeIfNecessary();
        int internalAttId = this.resolveInternalEccairsId(attributeId);
        TaxonomyServiceResponse attribute = this.getResponse(this.taxonomyServiceUrl + "/attributes/public/byID/" + internalAttId + "?taxonomyId=" + this.taxonomyVersion.id());
        assert (attribute != null);
        try {
            return (Integer)JsonPath.parse((String)attribute.getData().toString()).read("$.attributeValueList.levels", Integer.class, new Predicate[0]) > 1;
        }
        catch (PathNotFoundException e) {
            LOG.trace("Attribute {} does not have a value list.", (Object)attributeId);
            return false;
        }
    }

    private int resolveInternalEccairsId(int attributeId) {
        assert (this.taxonomyTree != null);
        if (this.attributeIdMap.containsKey(attributeId)) {
            return this.attributeIdMap.get(attributeId);
        }
        List attIds = (List)this.taxonomyTree.read("$..[?(@.tc==" + attributeId + " && @.type==\"A\")].id", (TypeRef)new TypeRef<List<Integer>>(){});
        if (attIds.isEmpty()) {
            throw new IllegalArgumentException("Attribute with ECCAIRS ID '" + String.valueOf(attIds) + "' not found in the taxonomy tree!");
        }
        Integer attId = (Integer)attIds.get(0);
        this.attributeIdMap.put(attributeId, attId);
        LOG.trace("Internal ECCAIRS ID of attribute {} is {}.", (Object)attributeId, (Object)attId);
        return attId;
    }

    public List<EccairsValue> getValueList(int attributeId) {
        LOG.trace("Loading value list of attribute {}.", (Object)attributeId);
        this.initializeIfNecessary();
        ArrayList<EccairsValue> result = new ArrayList<EccairsValue>();
        int attId = this.resolveInternalEccairsId(attributeId);
        TaxonomyServiceResponse topLevel = this.getResponse(this.taxonomyServiceUrl + "/attributes/public/showFirstLevelValues?attributesList=" + attId);
        topLevel.getData().get("map").get(Integer.toString(attId)).forEach(v -> {
            EccairsValue ev = this.initEccairsValue((JsonNode)v);
            result.add(ev);
            if (v.get("hasChild") != null && v.get("hasChild").asBoolean()) {
                ev.setValues(this.getValueDescendants(attributeId, v.get("id").intValue(), 2));
            }
        });
        return result;
    }

    private EccairsValue initEccairsValue(JsonNode valueNode) {
        EccairsValue ev = new EccairsValue();
        ev.setId(valueNode.get("identifier").intValue());
        ev.setDescription(valueNode.get("description").asText());
        ev.setDetailedDescription(valueNode.get("detailed").asText());
        ev.setLevel(valueNode.get("level").asText());
        ev.setExplanation(valueNode.get("explanation").asText());
        return ev;
    }

    private List<EccairsValue> getValueDescendants(int attributeId, int valId, int level) {
        LOG.trace("Loading value list of attribute {}, level {}.", (Object)attributeId, (Object)level);
        ArrayList<EccairsValue> result = new ArrayList<EccairsValue>();
        TaxonomyServiceResponse children = this.getResponse(this.taxonomyServiceUrl + "/listofvalue/public/childrenLov/" + valId);
        children.getData().get("list").forEach(v -> {
            EccairsValue ev = this.initEccairsValue((JsonNode)v);
            result.add(ev);
            if (v.get("hasChild") != null && v.get("hasChild").asBoolean()) {
                ev.setValues(this.getValueDescendants(attributeId, v.get("id").intValue(), level + 1));
            }
        });
        return result;
    }

    public EccairsEntity getEntity(int entityId) {
        this.initializeIfNecessary();
        List nodes = (List)this.taxonomyTree.read("$..[?(@.tc==" + entityId + " && @.type==\"E\")]", new Predicate[0]);
        if (nodes.isEmpty()) {
            throw new TaxonomyServiceException("Entity with id " + entityId + " not found in the taxonomy tree!");
        }
        Map node = (Map)nodes.get(0);
        return new EccairsEntity((Integer)node.get("id"), entityId, node.get("name").toString(), node.get("xsd").toString());
    }

    public EccairsAttribute getAttribute(int attributeId) {
        this.initializeIfNecessary();
        List nodes = (List)this.taxonomyTree.read("$..[?(@.tc==" + attributeId + " && @.type==\"A\")]", new Predicate[0]);
        if (nodes.isEmpty()) {
            throw new TaxonomyServiceException("Attribute with id " + attributeId + " not found in the taxonomy tree!");
        }
        Map node = (Map)nodes.get(0);
        return new EccairsAttribute((Integer)node.get("id"), attributeId, node.get("name").toString(), node.get("xsd").toString());
    }

    public EccairsEntity getParentEntity(int attributeId) {
        LOG.trace("Loading parent entity of attribute {}.", (Object)attributeId);
        this.initializeIfNecessary();
        int attId = this.resolveInternalEccairsId(attributeId);
        int taxonomyVersionId = this.taxonomyVersion.id();
        String payload = JsonPath.parse(Map.of("attributeIdentifiers", List.of(Integer.valueOf(attId)), "taxonomyId", taxonomyVersionId)).jsonString();
        TaxonomyServiceResponse response = this.postRequest(this.taxonomyServiceUrl + "/attributes/public/byIDs", payload);
        assert (response.getData().isArray());
        JsonNode parentEntity = response.getData().get(0).get("parentEntity");
        int parentId = parentEntity.get("taxonomyCode").asInt();
        return this.getEntity(parentId);
    }

    private TaxonomyServiceResponse postRequest(String uri, String jsonPayload) {
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(uri)).header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofString(jsonPayload)).build();
            return this.attemptRequest(request, 0);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new TaxonomyServiceException("Unable to perform request.", e);
        }
    }

    public void reset() {
        LOG.debug("Resetting taxonomy service");
        this.taxonomyTree = null;
        this.taxonomyVersion = null;
    }

    private static void configureJsonPath() {
        Configuration.setDefaults((Configuration.Defaults)new Configuration.Defaults(){
            private final JsonProvider jsonProvider = new JacksonJsonProvider();
            private final MappingProvider mappingProvider = new JacksonMappingProvider();

            public JsonProvider jsonProvider() {
                return this.jsonProvider;
            }

            public MappingProvider mappingProvider() {
                return this.mappingProvider;
            }

            public Set<Option> options() {
                return EnumSet.noneOf(Option.class);
            }
        });
    }
}

